PROCEDURE Verify_Address IS
  -- Perform address standardization and verification with CLEAN_Address()
  -- NOTE: This procedure should be fired on WHEN-VALIDATE-ITEM on all address fields
  l_address_rec          CLEAN_Address.ADDRESS_REC;
  fv_return_code         VARCHAR2(10);
  fv_return_description  VARCHAR2(500);
BEGIN
  -- Only Verify if the trigger item is the same as the cursor item
  -- (ie. don't fire more than once for one update)
  if :system.trigger_item != :system.cursor_item then
    return;
  end if;

  -- Leave if the address is being overridden
  if NVL(:ADDR.verified_override_flag,0) = 1 then
    return;
  end if;
  
  -- Exit if we don't have the required fields to verify
  if    (    :ADDR.address_line_1 IS NULL
         and :ADDR.address_line_2 IS NULL
         and :ADDR.address_line_3 IS NULL
        )
     or (    (:ADDR.city IS NULL or :ADDR.state IS NULL)
         and (:ADDR.city IS NULL or (:ADDR.country_code IS NULL and :ADDR.country_name IS NULL))
         and :ADDR.postal_code IS NULL
        )
     then
    return;
  end if;
  
  --
  -- Prime the address data record with input
  --
  l_address_rec.company        := :ADDR.company_name;
  l_address_rec.address_line_1 := :ADDR.address_line_1;
  l_address_rec.address_line_2 := :ADDR.address_line_2;
  l_address_rec.address_line_3 := :ADDR.address_line_3;
  l_address_rec.city           := :ADDR.city;
  l_address_rec.state          := :ADDR.state;
  l_address_rec.postal_code    := :ADDR.postal_code;
  l_address_rec.country_code   := :ADDR.country_code;
  l_address_rec.country_name   := :ADDR.country_name;
  
  --
  -- Verify the address - an error will be returned in the return code and description
  --
  CLEAN_Address.Verify_Generic_Silent(
               f_address_rec         => l_address_rec
              ,fv_return_code        => fv_return_code
              ,fv_return_description => fv_return_description
              ,fn_number_address_lines => 4
              ,fb_generic_address_name => FALSE
              ,fb_generic_company      => FALSE
              ,fb_address_suite_same_line => FALSE
              );
  if fv_return_code IS NOT NULL then
    WARNING(fv_return_description);
    return;
  end if;
  
  -- Remove the denormalized city,state postal line if this is a US/Canada address
  if NVL(l_address_rec.country_code,'US') in ('US','CA') then
  	l_address_rec.address_line_5 := NULL;
  end if;
  
  --
  -- Retrieve the output from the verified address buffer
  --
  :ADDR.company_name        := l_address_rec.company;

  if NVL(l_address_rec.country_code, 'US') in ('US','USA') then
    :ADDR.address_line_1      := l_address_rec.address_line_1;
    :ADDR.address_line_2      := l_address_rec.address_line_2;
    :ADDR.address_line_3      := l_address_rec.address_line_3;
  else
  	-- International Address
  	-- Use standardized address components on return
  	:ADDR.address_line_1      := l_address_rec.Address;
  	:ADDR.address_line_2      := l_address_rec.Suite;
  end if;
  :ADDR.city                := l_address_rec.city;
  :ADDR.state               := l_address_rec.state;
  :ADDR.postal_code         := l_address_rec.postal_code;
  :ADDR.country_code		    := l_address_rec.country_code;
  :ADDR.country_name        := l_address_rec.country_name;
  -- Zip Plus 4
  :ADDR.postal_code_extended:= l_address_rec.postal_code_extended;
  :ADDR.county_name         := l_address_rec.county_name;
  :ADDR.county_code         := l_address_rec.county_code;
  :ADDR.verify_error_code   := l_address_rec.error_code;
  :ADDR.verify_error_description := l_address_rec.error_string;
  :ADDR.verify_status_code  := NULL;

  :ADDR.verified_date       := sysdate;
                               
  :ADDR.timezone_code       := NULL;
  if l_address_rec.Timezone_Code IS NOT NULL then
  	if l_address_rec.Timezone_String IS NULL then
      :ADDR.timezone_code       := l_address_rec.Timezone_Code||'-'||cln$lookup.Get_Timezone(l_address_rec.Timezone_Code);
    else
      :ADDR.timezone_code       := l_address_rec.Timezone_Code||'-'||l_address_rec.Timezone_String;
    end if;
  end if;
  :ADDR.latitude            := l_address_rec.Latitude;
  :ADDR.longitude           := l_address_rec.Longitude;

  -- reset the address error and status strings
  if l_address_rec.error_code IS NOT NULL
  	 and l_address_rec.error_string IS NULL then
    :ADDR.verify_error_description  := cln$Lookup.Get_Address_Error( l_address_rec.error_code  );
  else
  	-- Use the DPV Status in the address Error if the error code is NULL
    if NVL(l_address_rec.DPV_Status,'AA') NOT IN ('AA','BB','RR')
       and l_address_rec.error_code IS NULL then
      l_address_rec.error_code := l_address_rec.DPV_Status;
      l_address_rec.error_string := substr(
                  NVL(cln$lookup.Help_DPV_Status(l_address_rec.DPV_Status)
                     ,cln$lookup.Get_DPV_Status(l_address_rec.DPV_Status)
                     )
                     ,1,100);
      :ADDR.verify_error_code         := l_address_rec.error_code;
      :ADDR.verify_error_description  := l_address_rec.error_string;
    end if;
  end if;
  if l_address_rec.status_code IS NOT NULL then
    :ADDR.verify_status_code := substr(l_address_rec.status_code
                                      ||'-'||cln$Lookup.Get_Address_Status( l_address_rec.status_code )
                                      ,1,100);
  end if;
  -- Get the DPV Status
  if l_address_rec.DPV_status IS NOT NULL then
    :ADDR.DPV_status := substr(l_address_rec.DPV_status
                              ||'-'||cln$Lookup.Get_DPV_Status( l_address_rec.DPV_status )
                              ,1,100);
  end if;

  if l_address_rec.address_type_code IS NOT NULL then
    :ADDR.address_type_code := substr(l_address_rec.address_type_code
                                     ||'-'||cln$Lookup.Get_Address_Type( l_address_rec.address_type_code )
                                     ,1,50);
  end if;

  -- Mark fields as valid so they don't trigger another verivication  
  set_item_property('ADDR.COMPANY_NAME',   ITEM_IS_VALID, PROPERTY_TRUE);
  set_item_property('ADDR.ADDRESS_LINE_1', ITEM_IS_VALID, PROPERTY_TRUE);
  set_item_property('ADDR.ADDRESS_LINE_2', ITEM_IS_VALID, PROPERTY_TRUE);
  set_item_property('ADDR.ADDRESS_LINE_3', ITEM_IS_VALID, PROPERTY_TRUE);
  set_item_property('ADDR.CITY',           ITEM_IS_VALID, PROPERTY_TRUE);
  set_item_property('ADDR.STATE',          ITEM_IS_VALID, PROPERTY_TRUE);
  set_item_property('ADDR.POSTAL_CODE',    ITEM_IS_VALID, PROPERTY_TRUE);
  set_item_property('ADDR.COUNTRY_CODE',   ITEM_IS_VALID, PROPERTY_TRUE);
  set_item_property('ADDR.COUNTRY_NAME',   ITEM_IS_VALID, PROPERTY_TRUE);

END;
